"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.filter2 = void 0;
var conv2_1 = require("./conv2");
/**
 * Rotates a matrix 180deg.
 *
 * @example
 * 1 2 3 4  becomes:  8 7 6 5
 * 5 6 7 8            4 3 2 1
 *
 * @method rotate1802d
 * @param {Matrix} mx - The input matrix
 * @returns {Matrix} out - The rotated matrix
 * @private
 * @memberOf matlab
 * @since 0.0.2
 */
function rotate1802d(_a) {
    var ref = _a.data, width = _a.width, height = _a.height;
    var data = new Array(ref.length);
    for (var i = 0; i < height; i++) {
        for (var j = 0; j < width; j++) {
            data[i * width + j] = ref[(height - 1 - i) * width + width - 1 - j];
        }
    }
    return {
        data: data,
        width: width,
        height: height,
    };
}
/**
 * Given a matrix X and a two-dimensional FIR filter h, filter2 rotates your filter matrix 180
 * degrees to create a convolution kernel. It then calls conv2, the two-dimensional convolution
 * function, to implement the filtering operation.
 *
 * This method mimics Matlab's `filter2` method
 *
 * @method filter2
 * @param {Matrix} h - The FIR filter
 * @param {Matrix} X - The input matrix
 * @param string [shape='same'] - The convolution shape
 * @returns {Matrix} conv - The 2D convolution of X with h
 * @public
 * @memberOf matlab
 * @since 0.0.2
 */
function filter2(h, X, shape) {
    if (shape === void 0) { shape = 'same'; }
    return conv2_1.conv2(X, rotate1802d(h), shape);
}
exports.filter2 = filter2;
//# sourceMappingURL=filter2.js.map